document.addEventListener("DOMContentLoaded", () => {
  // ——— helpers ———
  const $ = (id) => document.getElementById(id);
  const safeText = (el, txt) => { if (el) el.textContent = txt; };
  const safeShow = (el, display) => { if (el) el.style.display = display; };
  const safeAddEvent = (el, evt, handler) => { if (el) el.addEventListener(evt, handler); };

  function applyDonateTranslations(dict) {
    safeText($("online-text"), dict.online_players || "Online:");

    if ($("charname")) $("charname").placeholder = dict.placeholder_charname || "Enter your character name";
    if ($("trc20-input")) $("trc20-input").placeholder = dict.placeholder_trc20 || "Your TRC20 Wallet Address";
    if ($("discount-info")) $("discount-info").title = dict.tooltip_discount || "Spend more to unlock bigger discounts";

    safeText($("donate-title"), dict.donate_title || "Donate to Support the Server");
    safeText($("donate-description"), dict.donate_description || "Choose currency, enter character name and amount.");

    safeText($("btn-paypal-text"), dict.paypal_button || "Pay with PayPal");
    safeText($("btn-usdt-text"),   dict.usdt_button   || "Pay with USDT");
    safeText($("btn-stripe-text"), dict.stripe_button || "Pay with Stripe");

    safeText($("fee-paypal"),  dict.paypal_fee || "+5% fee");
    safeText($("fee-usdt"),    dict.usdt_fee   || "+10% fee");
    safeText($("fee-stripe"),  dict.stripe_fee || "no additional fees");

    safeText($("discount-info"),   dict.no_discount || "No discount");
    safeText($("raw-eur-preview"), dict.base_value_prefix || "Approx. base value: 0.00 EUR");

    if ($("status-text")) $("status-text").innerHTML = `${dict.status || "Server Status"}: <b class="offline">ONLINE</b>`;
    safeText($("qr-title"), dict.payment_title || "Payment");
    safeText($("trc20-confirm"), dict.confirm || "Confirm");
    safeText($("copy-button"), dict.copy_wallet || "Copy Wallet Address");
  }

  let dict = {};

  // ——— QWebChannel bridge ———
  if (typeof qt === "undefined" || !qt.webChannelTransport) {
    console.error("QWebChannel transport not found");
    return;
  }

  new QWebChannel(qt.webChannelTransport, function (channel) {
    const bridge = channel.objects.bridge;
    if (!bridge) {
      console.error("Bridge not found");
      return;
    }

    // DOM refs (может чего-то не быть — это нормально)
    const trc20InputContainer = $("trc20-input-container");
    const trc20Input = $("trc20-input");
    const trc20ConfirmBtn = $("trc20-confirm");

    const currencySelect = $("currency");
    const nameInput = $("charname");
    const amountInput = $("amount");

    const paypalBtn = $("pay-paypal");
    const usdtBtn = $("pay-usdt");
    const stripeBtn = $("pay-stripe");

    const paypalAmount = $("paypal-amount");
    const usdtAmount   = $("usdt-amount");
    const stripeAmount = $("stripe-amount");

    const discountInfo = $("discount-info");
    const discountProgress = $("discount-progress");
    const rawEurPreview = $("raw-eur-preview");

    const qrModal = $("qr-modal");
    const qrCanvas = $("qr-code");
    const qrTitle = $("qr-title");
    const qrClose = document.querySelector(".qr-close");
    const copyBtn = $("copy-button");

    let currentPaymentLink = "";

    // ——— QR Modal controls ———
    if (qrClose) {
      qrClose.onclick = () => {
        safeShow(qrModal, "none");
        safeShow(trc20InputContainer, "none");
        safeShow(qrCanvas, "block");
        safeShow(copyBtn, "block");
      };
    }

    if (copyBtn) {
      copyBtn.onclick = () => {
        if (!currentPaymentLink) return;
        const isUSDT = qrTitle && qrTitle.textContent.includes("USDT");
        const textToCopy = currentPaymentLink;
        navigator.clipboard.writeText(textToCopy)
          .then(() => alert((isUSDT ? "Wallet" : "Payment link") + " copied to clipboard."))
          .catch(() => alert("Failed to copy."));
      };
    }

    function showQR(title, value) {
      safeText(qrTitle, title);
      safeShow(qrModal, "flex");
      currentPaymentLink = value;
      if (qrCanvas && typeof QRCode !== "undefined") {
        QRCode.toCanvas(qrCanvas, value, { width: 220 }, function (error) {
          if (error) console.error(error);
        });
      }
    }

    // ——— Characters cache ———
    let characterCache = {};
    if (bridge.get_characters_cache) {
      bridge.get_characters_cache().then(json => {
        try { characterCache = JSON.parse(json) || {}; }
        catch (e) { console.error("❌ Failed to parse characters-cache.json:", e); }
      });
    }

    // ——— Nick validation (guarded) ———
    if (nameInput) {
      nameInput.addEventListener("input", () => {
        const nick = nameInput.value.trim().toLowerCase();
        const found = Object.prototype.hasOwnProperty.call(characterCache, nick);
        nameInput.style.border = found ? "" : "1px solid red";
        if (paypalBtn) paypalBtn.disabled = !found;
        if (usdtBtn)   usdtBtn.disabled   = !found;
        if (stripeBtn) stripeBtn.disabled = !found;
      });
    }

    // ——— Price calculation ———
    function calculatePrices() {
      if (!amountInput || !currencySelect || !bridge.get_price_preview) return;
      const value = parseFloat(amountInput.value) || 0;
      const currency = currencySelect.value;

      bridge.get_price_preview(currency, value).then(data => {
        try {
          const res = JSON.parse(data);
          safeText(rawEurPreview, `Approx. base value: ${res.baseEUR} EUR`);
          if (discountProgress) discountProgress.style.width = `${res.progressPercent}%`;
          safeText(discountInfo, res.discountText);
          if (paypalAmount) paypalAmount.textContent = `${res.paypal} EUR${res.discountSuffix}`;
          if (usdtAmount)   usdtAmount.textContent   = `${res.usdt} USDT${res.discountSuffix}`;
          if (stripeAmount) stripeAmount.textContent = `${res.stripe} EUR${res.discountSuffix}`;
        } catch (e) {
          console.error("❌ Failed to parse price preview:", e);
        }
      });
    }

    // ——— Online counter (same as before, with guards) ———
    function updateOnline() {
      if (!bridge.get_online_json) return;
      bridge.get_online_json(function (data) {
        try {
          const parsed = typeof data === "string" ? JSON.parse(data) : data;
          let online = 0;
          for (let key in parsed) {
            if (parsed[key]?.ConnectStat === 1) online++;
          }

          const boosted = online + 0;
          const onlineEl = $("online-count");
          if (onlineEl) onlineEl.textContent = boosted;

          const statusIcon = $("status-icon");
          if (statusIcon) {
            let iconName = "icon_on0.png";
            if (boosted > 10 && boosted <= 50) iconName = "icon_on1.png";
            else if (boosted > 50 && boosted <= 100) iconName = "icon_on2.png";
            else if (boosted > 100 && boosted <= 200) iconName = "icon_on3.png";
            else if (boosted > 200) iconName = "icon_on4.png";
            statusIcon.src = `../images/${iconName}`;
          }

          const fill = $("online-fill");
          if (fill) {
            const maxPlayers = 500;
            const percent = Math.min((boosted / maxPlayers) * 100, 100);
            fill.style.width = `${percent}%`;
            if (boosted <= 10) fill.style.backgroundColor = "#555";
            else if (boosted <= 50) fill.style.backgroundColor = "#4caf50";
            else if (boosted <= 100) fill.style.backgroundColor = "#2196f3";
            else if (boosted <= 200) fill.style.backgroundColor = "#ffc107";
            else fill.style.backgroundColor = "#f44336";
          }
        } catch (e) {
          console.error("❌ Failed to parse memb_stat.json:", e);
        }
      });
    }

    // ——— Validate inputs ———
    function validateFields() {
      const name = nameInput ? nameInput.value.trim() : "";
      const amount = amountInput ? parseFloat(amountInput.value) : NaN;
      const currency = currencySelect ? currencySelect.value : "";

      return new Promise(resolve => {
        if (!name || !amount || amount <= 0) {
          showError(dict.error_invalid_fields || "Please enter a valid character name and amount.");
          return resolve(false);
        }

        if (!bridge.get_price_preview || !currencySelect) {
          return resolve({ name, amount, currency });
        }

        bridge.get_price_preview(currency, amount).then(data => {
          try {
            const res = JSON.parse(data);
            const baseEUR = parseFloat(res.baseEUR);

            if (isNaN(baseEUR) || baseEUR < 1) {
              showError(dict.error_min_purchase || "Minimum purchase amount is equivalent to 1.00 EUR.");
              return resolve(false);
            }

            resolve({ name, amount, currency });
          } catch (e) {
            console.error("❌ Failed to parse price preview:", e);
            showError("Unexpected error while validating input.");
            resolve(false);
          }
        });
      });
    }

    function showError(text) {
      const info = $("payment-info");
      const error = $("error-message");
      if (!info || !error) return;
      safeShow(info, "none");
      safeText(error, text);
      safeShow(error, "block");

      setTimeout(() => {
        safeShow(error, "none");
        safeShow(info, "flex");
      }, 3000);
    }

    // ——— Click handlers (вешаем ТОЛЬКО если элемент существует) ———
    if (paypalBtn) {
      paypalBtn.addEventListener("click", () => {
        validateFields().then(fields => {
          if (!fields) return;
          safeShow(trc20InputContainer, "none");
          safeShow(qrCanvas, "block");
          safeShow(copyBtn, "block");

          if (bridge.get_paypal_link) {
            bridge.get_paypal_link(fields.name, fields.currency, fields.amount).then(url => {
              showQR((dict.scan_paypal || "Scan to Pay via PayPal"), url);
            });
          }
        });
      });
    }

    if (usdtBtn) {
      usdtBtn.addEventListener("click", () => {
        validateFields().then(fields => {
          if (!fields) return;

          // UI для ввода TRC20
          safeText(qrTitle, dict.enter_trc20 || "TRC20 Wallet");
          safeShow(trc20InputContainer, "block");
          safeShow(qrCanvas, "none");
          safeShow(copyBtn, "none");
          safeShow(qrModal, "flex");
          if (trc20Input) trc20Input.value = "";

          if (trc20ConfirmBtn) {
            trc20ConfirmBtn.onclick = null;
            trc20ConfirmBtn.onclick = () => {
              const trcAddress = trc20Input ? trc20Input.value.trim() : "";
              if (!trcAddress || !trcAddress.startsWith("T")) {
                alert("Please enter a valid TRC20 Wallet Address starting with 'T'.");
                return;
              }

              if (!bridge.save_trc20_address) return;
              bridge.save_trc20_address(fields.name, fields.currency, trcAddress, fields.amount).then(success => {
                if (success) {
                  safeShow(trc20InputContainer, "none");
                  safeText(qrTitle, dict.send_usdt || "Send USDT Payment");
                  safeShow(qrCanvas, "block");
                  safeShow(copyBtn, "block");
                  currentPaymentLink = trcAddress;
                  if (qrCanvas && typeof QRCode !== "undefined") {
                    QRCode.toCanvas(qrCanvas, trcAddress, { width: 220 }, error => {
                      if (error) console.error(error);
                    });
                  }
                } else {
                  alert("Failed to save TRC20 address. Please try again.");
                }
              });
            };
          }
        });
      });
    }

    if (stripeBtn) {
      stripeBtn.addEventListener("click", () => {
        validateFields().then(fields => {
          if (!fields) return;
          if (bridge.get_stripe_checkout_link) {
            bridge.get_stripe_checkout_link(fields.name, fields.currency, fields.amount).then(link => {
              showQR((dict.scan_stripe || "Complete payment via Stripe"), link);
            });
          }
        });
      });
    }

    // ——— Live bindings ———
    safeAddEvent(currencySelect, "change", calculatePrices);
    safeAddEvent(amountInput, "input", calculatePrices);
    calculatePrices(); // initial

    // ——— Online once at start ———
    updateOnline();

    // === 🌐 Translation (same API shape as в твоём файле)
    const lang = localStorage.getItem("language") || "en";
    if (bridge.get_translation) {
      bridge.get_translation(lang, raw => {
        try { dict = JSON.parse(raw) || {}; applyDonateTranslations(dict); }
        catch (e) { console.error("❌ Failed to parse translation:", e); }
      });
    }

    window.addEventListener("languageChanged", () => {
      const newLang = localStorage.getItem("language") || "en";
      if (bridge.get_translation) {
        bridge.get_translation(newLang, raw => {
          try { dict = JSON.parse(raw) || {}; applyDonateTranslations(dict); }
          catch (e) { console.error("❌ Failed to apply translation:", e); }
        });
      }
    });
  });
});
